/**
 * UniFi Device Search tool
 *
 * Copyright (c) 2020, Art of WiFi, info@artofwifi.net
 *
 * This source file is part of the UniFi Device Search tool and is subject to the MIT license that is bundled
 * with this package in the file LICENSE.md.
 *
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */

/**
 * declare some other vars which we will need later on
 */
var site_index             = 0,
    sites_array            = [],
    uap_count              = 0,
    usw_count              = 0,
    ugw_count              = 0,
    pending_device_count   = 0,
    total_device_count     = 0,
    nanobar                = {},
    uap_table              = {},
    usw_table              = {},
    ugw_table              = {},
    pending_adoption_table = {},
    all_devices_table      = {},
    filter_columns         = [0,3,5,7,8],
    controller             = {},
    controller_stats_loop;

/**
 * catch and process the selection of a UniFi controller
 */
$('.controller_idx').on('click', function(){
    var new_controller_idx = $(this).data('idx');
    if (!$(this).hasClass('active')) {
        /**
         * update the current controller idx
         */
        controller.idx       = new_controller_idx;
        controller.full_name = $(this).data('value');

        /**
         * we also cancel the loop for fetching the stats
         */
        clearInterval(controller_stats_loop);
        controller_stats_loop = null;

        /**
         * hide the device tables
         */
        $('#device_tables_wrapper').addClass('d-none');

        /**
         * clear the selected collection and the active options in the menu
         */
        selected_collection = {};
        $('.collection_idx').removeClass('active').children('i').remove();
        $('.dropdown-toggle').removeClass('active');

        /**
         * change the label in the controller dropdown "button"
         */
        $('#navbar_controller_dropdown_link').html(controller.full_name);

        /**
         * toggle the active class together with check marks
         */
        $('.controller_idx').removeClass('active').children('i').remove();
        $($(this)).addClass('active').append(' <i class="fas fa-check"></i>');

        /**
         * if not yet hidden, we hide all alerts and spinners
         */
        $('#select_controller_alert_wrapper').hide();
        $('.alert_wrapper').addClass('d-none');
        $('#controller_stats_placeholder').html('Fetching device and client totals');
        $('#controller_stats_spinner').addClass('d-none');
        $('#fetch_devices_button_wrapper').removeClass('d-none');
        $('#fetch_devices_button_wrapper').hide();

        /**
         * we also update the PHP $_SESSION variable with the new theme name using AJAX POST
         */
        $.ajax({
            type:     'POST',
            url:      'ajax/update_controller.php',
            dataType: 'json',
            data: {
                new_controller_idx: controller.idx
            },
            success:  function (json) {
                console.log('Switched to controller id: ' + controller.idx);
                $('.alert_wrapper').addClass('d-none');
                fetchControllerStats();
            },
            error: function(jqXHR, textStatus, errorThrown) {
                console.log(jqXHR);
                controller.idx       = '';
                controller.full_name = '';
            }
        });
    }
});

/**
 * fetch our client and device counts for all sites combined
 */
function fetchControllerStats() {
    console.log('Fetching top row totals');
    $('#controller_stats_wrapper').removeClass('d-none');
    $('#controller_stats_spinner').removeClass('d-none');
    $('#general_error').html('');

    fetchDebugDetails();

    /**
     * get the controller stats using AJAX
     */
    $.ajax({
        type: 'POST',
        url:  'ajax/fetch_controller_stats.php',
        data: {},
        dataType: 'json',
        success: function (json) {
            if (json.state === 'success') {
                onControllerStatsReceived(json)
            } else {
                console.log(json.message);
                controller.stats = {};
                renderGeneralErrorAlert(json.message);
                $('#controller_stats_placeholder').html('');
                $('#controller_stats_spinner').addClass('d-none');
            }
        },
        error: function(jqXHR, textStatus, errorThrown) {
            controller.stats = {};
            renderGeneralErrorAlert('AJAX response (' + textStatus + ') fetching stats: ' + errorThrown);
            $('#controller_stats_placeholder').html('');
            $('#controller_stats_spinner').addClass('d-none');
        }
    });
}

/**
 * callback function to process the controller stats received
 */
function onControllerStatsReceived(json) {
    controller.stats = json.stats;
    sites_array      = json.sites;

    /**
     * update the element with the controller stats
     */
    var stats = 'Sites: <b>' + controller.stats.site_count + '</b>' +
                ' | Access Points: <b>' + controller.stats.uap_count + '</b>' +
                ' | Switches: <b>' + controller.stats.usw_count + '</b>' +
                ' | Gateways: <b>' + controller.stats.ugw_count + '</b>' +
                ' | Pending adoption: <b>' + controller.stats.pending_devices_count + '</b>' +
                ' | WLAN users/guests: <b>' + controller.stats.wlan_users_count + '/' + controller.stats.wlan_guests_count + '</b>' +
                ' | LAN users/guests: <b>' + controller.stats.lan_users_count + '/' + controller.stats.lan_guests_count + '</b>';

    $('#controller_stats_placeholder').html(stats);
    $('#controller_stats_spinner').addClass('d-none');
    $('#fetch_devices_button_wrapper').show();
}

/**
 * catch a click on the fetch device details button:
 * - show device tables container
 * - process sites
 */
$('#fetch_devices_button').click(function(e) {
    e.preventDefault();
    if (!$('#fetch_devices_button').hasClass('disabled')) {
        $('#fetch_devices_button_icon').addClass('fa-spin');
        $('#fetch_devices_button').blur();
        $('#fetch_devices_button').addClass('disabled');

        $('#device_tables_wrapper').removeClass('d-none');

        /**
         * we also cancel the loop for fetching the stats
         */
        clearInterval(controller_stats_loop);
        controller_stats_loop = null;

        /**
         * empty the tables, the select2 elements and reset the counters in the tables header
         */
        $('#access_point_count').html('');
        $('#switch_count').html('');
        $('#usg_count').html('');
        $('#total_device_count').html('');
        $('#pending_device_count').html('');

        clearSelect2Lists(uap_table);
        clearSelect2Lists(usw_table);
        clearSelect2Lists(ugw_table);
        clearSelect2Lists(pending_adoption_table);

        if (enable_all_devices_table) {
            clearSelect2Lists(all_devices_table);
        }

        $.fn.dataTable.tables({api: true}).clear();
        $.fn.dataTable.tables({api: true}).columns.adjust().draw();

        /**
         * we then show the device tables and hide the button
         */
        setTimeout(function(){
            $('#device_tables_wrapper').show();

            /**
             * hide the elements for the "all devices" panel if it is disabled in the config file
             */
            if (!enable_all_devices_table) {
                $('.all_devices_panel').hide();
            }

            /**
             * call the function to process devices for the current controller
             */
            processSites(sites_array);
        }, 500);
    }
});

/**
 * function to process all sites available
 */
function processSites(sites_array) {
    console.log('Processing ' + sites_array.length + ' sites for controller id: ' + controller.idx);

    /**
     * we have the sites available in an array, each structured as follows:
     * site.site_id
     * site.site_full_name
     *
     * we fetch devices for each site
     */
    if (sites_array.length > 0) {
        sites_array.forEach(function(site) {
            fetchDevices(site.site_id, site.site_full_name);
        });
    } else {
        $('#alert_placeholder').html('<div class="alert alert-danger" role="alert"><i class="fas fa-exclamation-circle fa-lg fa-fw"></i> error fetching sites, none found</div>');
    }
}

/**
 * function to construct the AJAX request which fetches all the devices and their details for a site
 */
function fetchDevices(site_id, site_full_name) {
    if (debug) {
        console.time('fetchDevices-' + site_id);
    }

    /**
     * add the AJAX request to the queue
     */
    addAjax({
        type: 'POST',
        url:  'ajax/fetch_devices.php',
        data: {
            site_id:        site_id,
            site_full_name: site_full_name
        },
        dataType: 'json',
        success:  onDevicesReceived,
        error: function(jqXHR, textStatus, errorThrown) {
            $('#alert_placeholder').html('<div class="alert alert-danger" role="alert"><i class="fas fa-exclamation-circle fa-lg fa-fw"></i> AJAX response (' + textStatus + ') fetching devices: ' + errorThrown + '</div>');
        }
    });
}

/**
 * callback function to process the received device details
 */
function onDevicesReceived(json) {
    if (debug) {
        console.timeEnd('fetchDevices-' + json.site.site_id);
    }

    uap_count         += json.data.aps.length;
    usw_count         += json.data.usws.length;
    ugw_count         += json.data.usgs.length;
    total_device_count = uap_count + usw_count + ugw_count;

    $('#access_point_count').html('(' + uap_count + ')');
    $('#switch_count').html('(' + usw_count + ')');
    $('#usg_count').html('(' + ugw_count + ')');
    $('#total_device_count').html('(' + total_device_count + ')');

    /**
     * populate the array of new pending devices
     */
    var pending_devices_to_add = [];
    json.data.pending_adoption.forEach(function(pending_device) {
        var matches = pending_adoption_table.column(2).data().indexOf(pending_device.mac);
        if (matches < 0) {
            pending_devices_to_add.push(pending_device);
        }
    });

    pending_device_count += pending_devices_to_add.length;
    $('#pending_device_count').html('(' + pending_device_count + ')');

    /**
     * push data to each of the DataTables instances
     */
    usw_table.rows.add(json.data.usws);
    ugw_table.rows.add(json.data.usgs);
    uap_table.rows.add(json.data.aps);
    pending_adoption_table.rows.add(pending_devices_to_add);

    if (enable_all_devices_table) {
        all_devices_table.rows.add(json.data.usws);
        all_devices_table.rows.add(json.data.usgs);
        all_devices_table.rows.add(json.data.aps);
    }

    /**
     * redraw the visible tables
     */
    $.fn.dataTable.tables({visible: true, api: true}).columns.adjust().draw();

    /**
     * increment this var to keep track on progress and update
     * elements as required
     */
    site_index++;
    nanobar.go((site_index / controller.stats.site_count) * 100);
}

/**
 * function which handles the queueing and throttling of AJAX requests
 */
function addAjax(request) {
    if (debug) {
        console.log('Adding AJAX request to the queue for ' + request.data.site_id);
    }

    ajax_requests++;
    var old_success = request.success;
    var old_error   = request.error;
    var callback   = function() {
        ajax_requests--;

        /**
         * only if we are within our max concurrent requests and the queue is not empty
         * do we execute a request from the queue
         */
        if (ajax_active === ajax_max_conc && ajax_queue.length > 0) {
            $.ajax(ajax_queue.shift());
        } else {
            ajax_active--;
        }
    }

    request.success = function(resp, xhr, status) {
        callback();
        if (old_success) old_success(resp, xhr, status);
    };

    request.error = function(xhr, status, error) {
        callback();
        if (old_error) old_error(xhr, status, error);
    };

    if (ajax_active === ajax_max_conc) {
        ajax_queue.push(request);
    } else {
        ajax_active++;
        $.ajax(request);
    }
}

/**
 * After all the AJAX calls are done we do this
 */
$(document).ajaxStop(function() {
    /**
     * check whether we are now ready fetching all device details
     */
    if (typeof controller.stats !== 'undefined' && typeof controller.stats.site_count !== 'undefined' && site_index == controller.stats.site_count) {
        /**
         * reset the reload button icon and reset various vars
         */
        $('#fetch_devices_button_icon').removeClass('fa-spin');
        $('#fetch_devices_button').removeClass('disabled');
        /**
         * reset the reload button icon and reset various vars
         */
        $('#fetch_devices_button_icon').removeClass('fa-spin');
        $('#fetch_devices_button').removeClass('disabled');

        console.log('Generating column filters values');
        generateSelect2Lists(uap_table);
        generateSelect2Lists(usw_table);
        generateSelect2Lists(ugw_table);
        generateSelect2Lists(pending_adoption_table);

        if (enable_all_devices_table) {
            generateSelect2Lists(all_devices_table);
        }

        $("#progress_bar").html('').css('width', '0%').attr('aria-valuenow', 0);

        site_index           = 0;
        uap_count            = 0;
        usw_count            = 0;
        ugw_count            = 0;
        pending_device_count = 0;
        total_device_count   = 0;

        console.log('Tables ready!');
    }

    /**
     * reset our AJAX queue parameters again in case we receive a request
     * to reload the table data
     */
    ajax_requests = 0;
    ajax_queue    = [];
    ajax_active   = 0;

    /**
     * if not already active, we start the loop for fetching controller stats
     */
    if (typeof controller_stats_loop != 'number'){
        controller_stats_loop = setInterval(fetchControllerStats, totals_fetch_interval);
    }
});

/**
 * setting up the Datatables configuration template for all tables which will be modified as necessary
 */
var table_options = {
    dom:       "<'row'<'col-sm-6'l><'col-sm-6'f>>" +
               "<'row'<'col-sm-12'tr>>" +
               "<'row'<'col-sm-5'i><'col-sm-7'p>>" +
               "<'row'<'col-sm-12'B>>",
    autoWidth: true,
    stateSave: true,
    scrollX:   true,
    scrollY:   false,
    order: [
        [0, 'asc']
    ],
    buttons: {
        buttons: [
            {
                extend:            'colvis',
                text:              '<i class="fa fa-columns" aria-hidden="true"></i> Select columns',
                className:         'btn btn-info',
                dropup:            true,
                collectionLayout: 'two-column'
            },
            {
                text:      '<i class="fa fa-times" aria-hidden="true"></i> Reset all filters',
                className: 'btn btn-info',
                action:    function (e, dt, node, config ) {
                    dt.search('').draw();
                    resetSelect2Filters(dt);
                }
            }
        ],
        dom: {
            button: {
                className: ''
            }
        }
    },
    columns: [
        {data: 'site_name'},
        {data: 'name'},
        {
            data:  'mac',
            width: '125px'
        },
        {
            data:   'model',
            render: modelRenderFunction,
            width:  '200px'
        },
        {
            data:  'ip',
            width: '100px'
        },
        {
            data:  'version',
            width: '100px',
            type:  'natural'
        },
        {
            data:  'serial',
            width: '100px'
        },
        {
            data:   'state',
            render: stateRenderFunction,
            width:  '75px'
        },
        {
            data:   'is_lts',
            render: supportRenderFunction,
            width:  '75px'
        },
        {
            data:   'eol_date'
        },
        {
            data:    'ssids',
            render: function(data, type, full, meta) {
                if (type === 'display' || type === 'filter') {
                    if (typeof data === 'undefined' || data == null || data == 0) {
                        return '';
                    }

                    return data;
                }
                return data;
            }
        }
    ],
    initComplete: function () {
        /**
         * we need this to create unique ids for the select2 elements
         */
        var container_id = $(this.api().table().container()).attr('id');

        /**
         * only filter on specified columns
         */
        this.api().columns(filter_columns).every(function () {
            var title = this.header();

            /**
             * replace spaces with dashes
             */
            title = $(title).html().replace(/[\W]/g, '-');
            var column = this;
            var select = $('<select id="' + container_id + '-' + title + '" class="select2 ' + container_id + '-' + title + '-select2' + '"></select>')
                .appendTo($(column.footer()).empty())
                .on('change', function () {
                    /**
                     * Get the "text" property from each selected data regex escape the value and store in array
                     */
                    var data = $.map($(this).select2('data'), function( value, key ) {
                        return value.text ? '^' + $.fn.dataTable.util.escapeRegex(value.text) + '$' : null;
                    });

                    /**
                     * if no data selected use ""
                     */
                    if (data.length === 0) {
                        data = [''];
                    }

                    /**
                     * join array into string with regex or (|)
                     */
                    var val = data.join('|');

                    /**
                     * search for the option(s) selected
                     */
                    column.search(val ? val : '', true, false).draw();
                } );

            /**
             * use column title as selector
             */
            $('#' + container_id + '-' + title).select2({
                theme:         'bootstrap',
                width:         '100%',
                multiple:      true,
                closeOnSelect: true,
                placeholder:   'All'
            });

            /**
             * initially clear select otherwise first option is selected
             */
            $('#' + container_id + '-' + title).val(null).trigger('change');
        });
    }
}

function generateSelect2Lists(this_table) {
    /**
     * we need this to create unique ids for the select2 elements
     */
    var container_id = $(this_table.table().container()).attr('id');

    if (debug) {
        console.time('generateOptions-' + container_id);
    }

    /**
     * only filter on specified columns
     */
    this_table.columns(filter_columns).every(function () {
        var title = this.header();

        /**
         * replace spaces with dashes
         */
        title = $(title).html().replace(/[\W]/g, '-');
        var column = this;
        var select = $('.' + container_id + '-' + title + '-select2');

        /**
         * generate an array of non-empty, unique options
         */
        this_table.cells(null, column.index())
            .render('filter')
            .unique()
            .sort(function(a, b) {
                return naturalSort(a, b);
            })
            .each(function (d, j) {
                if (d !== '') {
                    var newOption = new Option(d, d, false, false);
                    select.append(newOption);
                }
            });

        /**
         * initially clear select otherwise first option is selected
         */
        select.val(null).trigger('change');
    });

    if (debug) {
        console.timeEnd('generateOptions-' + container_id);
    }
}

function clearSelect2Lists(this_table) {
    /**
     * we need this to create unique ids for the select2 elements
     */
    var container_id = $(this_table.table().container()).attr('id');

    /**
     * only filter on specified columns
     */
    this_table.columns(filter_columns).every(function () {
        var title = this.header();

        /**
         * replace spaces with dashes
         */
        title = $(title).html().replace(/[\W]/g, '-');
        var column = this;
        var select = $('.' + container_id + '-' + title + '-select2');

        select.empty();
    });
}

function resetSelect2Filters(this_table) {
    /**
     * we need this to create unique ids for the select2 elements
     */
    var container_id = $(this_table.table().container()).attr('id');

    /**
     * only filter on specified columns
     */
    this_table.columns(filter_columns).every(function () {
        var title = this.header();

        /**
         * replace spaces with dashes
         */
        title = $(title).html().replace(/[\W]/g, '-');
        var column = this;
        var select = $('.' + container_id + '-' + title + '-select2');

        select.val(null).trigger('change');
    });
}

/**
 * configure the DataTables instance for the Access Points table
 */
uap_table = $('#uap_table').DataTable(table_options);
uap_table.columns([9, 10]).visible(false);

device_group  = 'Access Points';
uap_table.button().add(2, {
    extend:    'csv',
    text:      '<i class="fa fa-download" aria-hidden="true"></i> CSV export',
    className: 'btn btn-primary',
    exportOptions: {
        columns: ':visible'
    },
    filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
});

uap_table.button().add(3, {
    extend:    'excel',
    text:      '<i class="fa fa-download" aria-hidden="true"></i> Excel export',
    className: 'btn btn-success',
    exportOptions: {
        columns: ':visible'
    },
    title:     'Unifi Device Search tool - ' + device_group + ' export (create time: ' + moment().format('DD-MM-YYYY HH:mm') + ')',
    filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
});

/**
 * change the columns config by removing the SSID column (last object in the columns array)
 */
table_options.columns.pop();

/**
 * configure the DataTables instance for the Switches table
 */
usw_table = $('#usw_table').DataTable(table_options);
usw_table.column(9).visible(false);

device_group  = 'Switches';
usw_table.button().add(2, {
    extend:    'csv',
    text:      '<i class="fa fa-download" aria-hidden="true"></i> CSV export',
    className: 'btn btn-primary',
    exportOptions: {
        columns: ':visible'
    },
    filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
});

usw_table.button().add(3, {
    extend:    'excel',
    text:      '<i class="fa fa-download" aria-hidden="true"></i> Excel export',
    className: 'btn btn-success',
    exportOptions: {
        columns: ':visible'
    },
    title:     'Unifi Device Search tool - ' + device_group + ' export (create time: ' + moment().format('DD-MM-YYYY HH:mm') + ')',
    filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
});

/**
 * configure the DataTables instance for the USGs table
 */
ugw_table = $('#ugw_table').DataTable(table_options);
ugw_table.column(9).visible(false);

device_group = 'USGs';
ugw_table.button().add(2, {
    extend:    'csv',
    text:      '<i class="fa fa-download" aria-hidden="true"></i> CSV export',
    className: 'btn btn-primary',
    exportOptions: {
        columns: ':visible'
    },
    filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
});

ugw_table.button().add(3, {
    extend:    'excel',
    text:      '<i class="fa fa-download" aria-hidden="true"></i> Excel export',
    className: 'btn btn-success',
    exportOptions: {
        columns: ':visible'
    },
    title:     'Unifi Device Search tool - ' + device_group + ' export (create time: ' + moment().format('DD-MM-YYYY HH:mm') + ')',
    filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
});

/**
 * configure the DataTables instance for the Devices Pending Adoption table
 */
pending_adoption_table = $('#pending_devices_table').DataTable(table_options);
pending_adoption_table.columns([0,1,9]).visible(false);

device_group = 'Devices Pending Adoption';
pending_adoption_table.button().add(2, {
    extend:    'csv',
    text:      '<i class="fa fa-download" aria-hidden="true"></i> CSV export',
    className: 'btn btn-primary',
    exportOptions: {
        columns: ':visible'
    },
    filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
});

pending_adoption_table.button().add(3, {
    extend:    'excel',
    text:      '<i class="fa fa-download" aria-hidden="true"></i> Excel export',
    className: 'btn btn-success',
    exportOptions: {
        columns: ':visible'
    },
    title:     'Unifi Device Search tool - ' + device_group + ' export (create time: ' + moment().format('DD-MM-YYYY HH:mm') + ')',
    filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
});

/**
 * configure the DataTables instance for the table containing all devices,
 * but only if the table has been enabled through the config file
 */
if (enable_all_devices_table) {
    all_devices_table = $('#all_devices_table').DataTable(table_options);
    ugw_table.column(9).visible(false);

    device_group = 'All UniFi devices';
    all_devices_table.button().add(2, {
        extend:    'csv',
        text:      '<i class="fa fa-download" aria-hidden="true"></i> CSV export',
        className: 'btn btn-primary',
        exportOptions: {
            columns: ':visible'
        },
        filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
    });

    all_devices_table.button().add(3, {
        extend:    'excel',
        text:      '<i class="fa fa-download" aria-hidden="true"></i> Excel export',
        className: 'btn btn-success',
        exportOptions: {
            columns: ':visible'
        },
        title:     'Unifi Device Search tool - ' + device_group + ' export (create time: ' + moment().format('DD-MM-YYYY HH:mm') + ')',
        filename:  'Unifi Device Search tool - ' + device_group + ' export - ' + moment().format('DD-MM-YYYY-HHmm')
    });
}

/**
 * add regex search capabilities for global filter input
 * https://datatables.net/examples/api/regex.html
 *
 * TODO:
 * consider whether to only apply filter to visible columns
 */
$("input[type='search']").keyup(function() {
    var table_name = $(this).attr('aria-controls');
    $('#' + table_name).DataTable()
                       .search(
                           $(this).val(),
                           use_regex_filter,
                           !use_regex_filter
                       )
                       .draw();
});

/**
 * fix to get Bootstrap tooltips to work within Datatables
 */
$('.table').on('draw.dt', function () {
    $('[data-toggle="tooltip"]').tooltip({
        container: 'body'
    });
});

/**
 * redraw the tables upon tab being opened
 */
$(document).on('shown.bs.tab', 'a[data-toggle="tab"]', function (e) {
    $.fn.dataTable.tables({visible: true, api: true}).columns.adjust().draw();
});

$(document).ready(function() {
    /**
     * initialise the nanobar progress bar
     */
    nanobar = new Nanobar({target: document.getElementById('tab_content')});
});
