/**
 * UniFi Device Search tool, common JavaScript functions
 *
 * Copyright (c) 2020, Art of WiFi, info@artofwifi.net
 *
 * This source file is part of the UniFi Device Search tool and is subject to the MIT license that is bundled
 * with this package in the file LICENSE.md.
 *
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */

/**
 * defaults for DataTables
 */
if (jQuery().DataTable) {
    $.fn.dataTable.ext.errMode = 'throw';

    jQuery.extend(jQuery.fn.dataTableExt.oSort, {
        'natural-asc': function (a, b) {
            return naturalSort(a,b,true);
        },
        'natural-desc': function (a, b) {
            return naturalSort(a, b, true) * -1;
        },
        'natural-nohtml-asc': function(a, b) {
            return naturalSort(a, b, false);
        },
        'natural-nohtml-desc': function(a, b) {
            return naturalSort(a, b, false) * -1;
        },
        'natural-ci-asc': function(a, b) {
            a = a.toString().toLowerCase();
            b = b.toString().toLowerCase();

            return naturalSort(a, b, true);
        },
        'natural-ci-desc': function(a, b) {
            a = a.toString().toLowerCase();
            b = b.toString().toLowerCase();

            return naturalSort(a, b, true) * -1;
        }
    });
}

/**
 * Natural Sort algorithm for Javascript - Version 0.7 - Released under MIT license
 * Author: Jim Palmer (based on chunking idea from Dave Koelle)
 * Contributors: Mike Grier (mgrier.com), Clint Priest, Kyle Adams, guillermo
 * See: http://js-naturalsort.googlecode.com/svn/trunk/naturalSort.js
 */
function naturalSort (a, b, html) {
    var re    = /(^-?[0-9]+(\.?[0-9]*)[df]?e?[0-9]?%?$|^0x[0-9a-f]+$|[0-9]+)/gi,
        sre   = /(^[ ]*|[ ]*$)/g,
        dre   = /(^([\w ]+,?[\w ]+)?[\w ]+,?[\w ]+\d+:\d+(:\d+)?[\w ]?|^\d{1,4}[\/\-]\d{1,4}[\/\-]\d{1,4}|^\w+, \w+ \d+, \d{4})/,
        hre   = /^0x[0-9a-f]+$/i,
        ore   = /^0/,
        htmre = /(<([^>]+)>)/ig,
        // convert all to strings and trim()
        x     = a.toString().replace(sre, '') || '',
        y     = b.toString().replace(sre, '') || '';

    /**
     * remove html from strings if desired
     */
    if (!html) {
        x = x.replace(htmre, '');
        y = y.replace(htmre, '');
    }

    /**
     * chunk/tokenize
     */
    var xN = x.replace(re, '\0$1\0').replace(/\0$/, '').replace(/^\0/, '').split('\0'),
        yN = y.replace(re, '\0$1\0').replace(/\0$/, '').replace(/^\0/, '').split('\0'),
        // numeric, hex or date detection
        xD = parseInt(x.match(hre), 10) || (xN.length !== 1 && x.match(dre) && Date.parse(x)),
        yD = parseInt(y.match(hre), 10) || xD && y.match(dre) && Date.parse(y) || null;

    /**
     * first try and sort Hex codes or Dates
     */
    if (yD) {
        if (xD < yD) {
            return -1;
        }
        else if (xD > yD) {
            return 1;
        }
    }

    /**
     * natural sorting through split numeric strings and default strings
     */
    for(var cLoc = 0, numS = Math.max(xN.length, yN.length); cLoc < numS; cLoc++) {
        // find floats not starting with '0', string or 0 if not defined (Clint Priest)
        var oFxNcL = !(xN[cLoc] || '').match(ore) && parseFloat(xN[cLoc], 10) || xN[cLoc] || 0;
        var oFyNcL = !(yN[cLoc] || '').match(ore) && parseFloat(yN[cLoc], 10) || yN[cLoc] || 0;

        /**
         * handle numeric vs string comparison - number < string - (Kyle Adams)
         */
        if (isNaN(oFxNcL) !== isNaN(oFyNcL)) {
            return (isNaN(oFxNcL)) ? 1 : -1;
        } else if (typeof oFxNcL !== typeof oFyNcL) {
            /**
             * rely on string comparison if different types - i.e. '02' < 2 != '02' < '2'
             */
            oFxNcL += '';
            oFyNcL += '';
        }

        if (oFxNcL < oFyNcL) {
            return -1;
        }

        if (oFxNcL > oFyNcL) {
            return 1;
        }
    }
    return 0;
}

/**
 * render function for Datatables "state" columns
 */
function stateRenderFunction(data, type, full, meta) {
    if (type === 'display') {
        let suffix = '';
        if (typeof full.disabled !== 'undefined' && full.disabled === true) {
            suffix = ' <span class="badge badge-warning">disabled</span>'
        }

        switch(data) {
            case 1:
                if (full.locating === true) {
                    return ('<span class="badge badge-primary">flashing</span>' + suffix);
                }

                return ('<span class="badge badge-success">OK</span>' + suffix);
                break;
            case 0:
                return ('<span class="badge badge-danger">offline</span>' + suffix);
                break;
            case 2:
                return ('<span class="badge badge-warning">pending adoption</span>' + suffix);
                break;
            case 4:
                return ('<span class="badge badge-warning">provisioning</span>' + suffix);
                break;
            case 5:
                return ('<span class="badge badge-warning">updating</span>' + suffix);
                break;
            case 6:
                return ('<span class="badge badge-danger">unreachable</span>' + suffix);
                break;
            case 7:
                return ('<span class="badge badge-warning">adopting</span>' + suffix);
                break;
            case 9:
                return ('<span class="badge badge-warning">adoption error</span>' + suffix);
                break;
            case 11:
                return ('<span class="badge badge-warning">isolated</span>' + suffix);
                break;
            default:
                return data;
        }
    }

    if (type === 'filter') {
        let suffix = '';
        if (typeof full.disabled !== 'undefined' && full.disabled === true) {
            suffix = '-disabled'
        }

        switch(data) {
            case 1:
                if (full.locating === true) {
                    return 'flashing' + suffix;
                }

                return 'OK' + suffix;
                break;
            case 0:
                return 'offline' + suffix;
                break;
            case 2:
                return 'pending adoption' + suffix;
                break;
            case 4:
                return 'provisioning' + suffix;
                break;
            case 5:
                return 'updating' + suffix;
                break;
            case 6:
                return 'unreachable' + suffix;
                break;
            case 7:
                return 'adopting' + suffix;
                break;
            case 9:
                return 'adoption error' + suffix;
                break;
            case 11:
                return 'isolated' + suffix;
                break;
            default:
                return data;
        }
    }

    return data;
}

/**
 * render function for Datatables "LTS/EOL" columns
 */
function supportRenderFunction(data, type, full, meta) {
    if (type === 'display') {
        switch(data) {
            case true:
                if (full.eol_date !== '') {
                    return ('<span class="badge badge-warning clickable" data-toggle="tooltip" data-placement="left" title="EOL date: ' + moment(full.eol_date).format('MMMM Do YYYY') + '">LTS</span>');
                }

                return ('<span class="badge badge-warning">LTS</span>');
                break;
            default:
                if (full.is_eol) {
                    if (full.eol_date !== '') {
                        return ('<span class="badge badge-danger clickable" data-toggle="tooltip" data-placement="left" title="EOL date: ' + moment(full.eol_date).format('MMMM Do YYYY') + '">EOL</span>');
                    }

                    return ('<span class="badge badge-danger">EOL</span>');
                }

                return '';
        }
    }

    if (type === 'filter') {
        switch(data) {
            case true:
                return 'LTS';
                break;
            default:
                if (full.is_eol) {
                    return 'EOL';
                }

                return '';
        }
    }

    return data;
}

/**
 * render function for Datatables column "model"
 */
function modelRenderFunction(data, type, full, meta) {
    if (type === 'display' || type === 'filter') {
        return getDeviceModelName(data);
    }

    return data;
}

/**
 * array of objects containing each device model we know of
 *
 * NOTE:
 * - needs to be updated when new models are released
 * - updated for 5.12.66 from /usr/lib/unifi/dl/firmware/bundles.json
 */
var device_models = {
    "BZ2": {
        "base_model": "BZ2",
        "display": "UniFi AP"
    },
    "BZ2LR": {
        "base_model": "BZ2",
        "display": "UniFi AP-LR"
    },
    "S216150": {
        "base_model": "US24P250",
        "display": "UniFi Switch 16 AT-150W",
        "include_fw": "USC8"
    },
    "S224250": {
        "base_model": "US24P250",
        "display": "UniFi Switch 24 AT-250W",
        "include_fw": "USC8"
    },
    "S224500": {
        "base_model": "US24P250",
        "display": "UniFi Switch 24 AT-500W",
        "include_fw": "USC8"
    },
    "S248500": {
        "base_model": "US24P250",
        "display": "UniFi Switch 48 AT-500W",
        "include_fw": "USC8"
    },
    "S248750": {
        "base_model": "US24P250",
        "display": "UniFi Switch 48 AT-750W",
        "include_fw": "USC8"
    },
    "S28150": {
        "base_model": "US24P250",
        "display": "UniFi Switch 8 AT-150W",
        "include_fw": "USC8"
    },
    "U2HSR": {
        "base_model": "U2HSR",
        "display": "UniFi AP-Outdoor+"
    },
    "U2IW": {
        "base_model": "U2IW",
        "display": "UniFi AP-In Wall"
    },
    "U2Lv2": {
        "base_model": "U2Sv2",
        "display": "UniFi AP-LR v2"
    },
    "U2O": {
        "base_model": "BZ2",
        "display": "UniFi AP-Outdoor"
    },
    "U2Sv2": {
        "base_model": "U2Sv2",
        "display": "UniFi AP v2"
    },
    "U5O": {
        "base_model": "BZ2",
        "display": "UniFi AP-Outdoor 5G"
    },
    "U7E": {
        "base_model": "U7E",
        "display": "UniFi AP-AC"
    },
    "U7EDU": {
        "base_model": "U7PG2",
        "display": "UniFi AP-AC-EDU"
    },
    "U7Ev2": {
        "base_model": "U7E",
        "display": "UniFi AP-AC v2"
    },
    "U7HD": {
        "base_model": "U7HD",
        "display": "UniFi AP-HD"
    },
    "U7IW": {
        "base_model": "U7PG2",
        "display": "UniFi AP-AC-In Wall"
    },
    "U7IWP": {
        "base_model": "U7PG2",
        "display": "UniFi AP-AC-In Wall Pro"
    },
    "U7LR": {
        "base_model": "U7PG2",
        "display": "UniFi AP-AC-LR"
    },
    "U7LT": {
        "base_model": "U7PG2",
        "display": "UniFi AP-AC-Lite"
    },
    "U7MP": {
        "base_model": "U7PG2",
        "display": "UniFi AP-AC-Mesh-Pro"
    },
    "U7MSH": {
        "base_model": "U7PG2",
        "display": "UniFi AP-AC-Mesh"
    },
    "U7NHD": {
        "base_model": "U7NHD",
        "display": "UniFi AP-nanoHD"
    },
    "U7O": {
        "base_model": "U7E",
        "display": "UniFi AP-AC Outdoor"
    },
    "U7P": {
        "base_model": "U7P",
        "display": "UniFi AP-Pro"
    },
    "U7PG2": {
        "base_model": "U7PG2",
        "display": "UniFi AP-AC-Pro"
    },
    "U7SHD": {
        "base_model": "U7HD",
        "display": "UniFi AP-SHD"
    },
    "UAE6": {
        "base_model": "UAE6",
        "display": "U6-Extender"
    },
    "UAIW6": {
        "base_model": "UAIW6",
        "display": "U6-IW"
    },
    "UAL6": {
        "base_model": "UAL6",
        "display": "U6-Lite"
    },
    "UALR6": {
        "base_model": "UALR6",
        "display": "U6-LR"
    },
    "UAM6": {
        "base_model": "UAM6",
        "display": "U6-Mesh"
    },
    "UAP6": {
        "base_model": "UAP6",
        "display": "U6-Pro"
    },
    "UBB": {
        "base_model": "UBB",
        "display": "UniFi Building Bridge"
    },
    "UCMSH": {
        "base_model": "U7HD",
        "display": "UniFi AP-MeshXG"
    },
    "UCXG": {
        "base_model": "U7HD",
        "display": "UniFi AP-XG"
    },
    "UDC48X6": {
        "base_model": "UDC48X6",
        "display": "UniFi Data Center 100G-48X6"
    },
    "UDM": {
        "base_model": "UDM",
        "display": "UniFi Dream Machine"
    },
    "UDMB": {
        "base_model": "U7NHD",
        "display": "UDM Beacon"
    },
    "UDMPRO": {
        "base_model": "UDMPRO",
        "display": "UniFi Dream Machine PRO"
    },
    "UDMSE": {
        "base_model": "UDMSE",
        "display": "UniFi Dream Machine SE"
    },
    "UFLHD": {
        "base_model": "U7NHD",
        "display": "UniFi Flex HD"
    },
    "UGW3": {
        "base_model": "UGW3",
        "display": "UniFi Security Gateway"
    },
    "UGW4": {
        "base_model": "UGW4",
        "display": "UniFi Security Gateway-Pro"
    },
    "UGWXG": {
        "base_model": "UGWXG",
        "display": "UniFi Security Gateway XG-8"
    },
    "UHDIW": {
        "base_model": "U7NHD",
        "display": "UniFi AP-HD-In Wall"
    },
    "ULTE": {
        "base_model": "ULTE",
        "display": "UniFi AP-LTE"
    },
    "ULTEPEU": {
        "base_model": "ULTEPEU",
        "display": "UniFi LTE Pro EU"
    },
    "ULTEPUS": {
        "base_model": "ULTEPUS",
        "display": "UniFi LTE Pro US"
    },
    "UP1": {
        "base_model": "UP1",
        "display": "UniFi Smart Power Plug"
    },
    "UP6": {
        "base_model": "UP6",
        "display": "UniFi Smart Power 6-Port Power Strip"
    },
    "US16P150": {
        "base_model": "US24P250",
        "display": "UniFi Switch 16 POE-150W",
        "include_fw": "USC8"
    },
    "US24": {
        "base_model": "US24P250",
        "display": "UniFi Switch 24",
        "include_fw": "USC8"
    },
    "US24P250": {
        "base_model": "US24P250",
        "display": "UniFi Switch 24 POE-250W",
        "include_fw": "USC8"
    },
    "US24P500": {
        "base_model": "US24P250",
        "display": "UniFi Switch 24 POE-500W",
        "include_fw": "USC8"
    },
    "US24PL2": {
        "base_model": "US24PL2",
        "display": "UniFi Switch 24 L2 POE"
    },
    "US24PRO": {
        "base_model": "US24PRO",
        "display": "UniFi Switch 24 PRO"
    },
    "US24PRO2": {
        "base_model": "US24PRO2",
        "display": "UniFi Switch 24 PRO"
    },
    "US48": {
        "base_model": "US24P250",
        "display": "UniFi Switch 48",
        "include_fw": "USC8"
    },
    "US48P500": {
        "base_model": "US24P250",
        "display": "UniFi Switch 48 POE-500W",
        "include_fw": "USC8"
    },
    "US48P750": {
        "base_model": "US24P250",
        "display": "UniFi Switch 48 POE-750W",
        "include_fw": "USC8"
    },
    "US48PL2": {
        "base_model": "US24PL2",
        "display": "UniFi Switch 48 L2 POE"
    },
    "US48PRO": {
        "base_model": "US48PRO",
        "display": "UniFi Switch 48 PRO"
    },
    "US48PRO2": {
        "base_model": "US48PRO2",
        "display": "UniFi Switch 48 PRO"
    },
    "US624P": {
        "base_model": "US624P",
        "display": "UniFi6 Switch 24"
    },
    "US6XG150": {
        "base_model": "US6XG150",
        "display": "UniFi Switch 6 XG"
    },
    "US8": {
        "aka_by": "USC8",
        "base_model": "US24P250",
        "display": "UniFi Switch 8",
        "include_fw": "USC8"
    },
    "US8P150": {
        "aka_by": "USC8P150",
        "base_model": "US24P250",
        "display": "UniFi Switch 8 POE-150W",
        "include_fw": "USC8"
    },
    "US8P60": {
        "aka_by": "USC8P60",
        "base_model": "US24P250",
        "display": "UniFi Switch 8 POE-60W",
        "include_fw": "USC8"
    },
    "USC8": {
        "base_model": "USC8P450",
        "display": "UniFi Switch 8",
        "hide": true,
        "include_fw": "US24P250"
    },
    "USC8P150": {
        "base_model": "USC8P450",
        "display": "UniFi Switch 8 POE-150W",
        "hide": true
    },
    "USC8P450": {
        "base_model": "USC8P450",
        "display": "UniFi Industrial Switch 8 POE-450W"
    },
    "USC8P60": {
        "base_model": "USC8P450",
        "display": "UniFi Switch 8 POE-60W",
        "hide": true
    },
    "USF5P": {
        "base_model": "USF5P",
        "display": "UniFi Switch Flex 5 POE"
    },
    "USL16LP": {
        "base_model": "USL16LP",
        "display": "UniFi Switch Lite 16 POE"
    },
    "USL16P": {
        "base_model": "USL16P",
        "display": "UniFi Switch 16 POE"
    },
    "USL24": {
        "base_model": "USL24",
        "display": "UniFi Switch 24"
    },
    "USL24P": {
        "base_model": "USL24P",
        "display": "UniFi Switch 24 POE"
    },
    "USL48": {
        "base_model": "USL48",
        "display": "UniFi Switch 48"
    },
    "USL48P": {
        "base_model": "USL48P",
        "display": "UniFi Switch 48 POE"
    },
    "USL8A": {
        "base_model": "USL8A",
        "display": "USW-AGGREGATION"
    },
    "USL8LP": {
        "base_model": "USL8LP",
        "display": "UniFi Switch Lite 8 POE"
    },
    "USL8MP": {
        "base_model": "USL8MP",
        "display": "UniFi Switch Mission Critical"
    },
    "USMINI": {
        "base_model": "USMINI",
        "display": "Unifi Switch Mini"
    },
    "USPPDUP": {
        "base_model": "USPPDUP",
        "display": "UniFi Smart Power Power Distribution Unit Pro"
    },
    "USPRPS": {
        "base_model": "USF5P",
        "display": "UniFi Smart Power Redundant Power System"
    },
    "USXG": {
        "base_model": "USXG",
        "display": "UniFi Switch XG"
    },
    "UXBSDM": {
        "base_model": "U7HD",
        "display": "UniFi AP-BlackBaseStationXG"
    },
    "UXGPRO": {
        "base_model": "UXGPRO",
        "display": "UniFi NeXt-Gen Gateway PRO"
    },
    "UXSDM": {
        "base_model": "U7HD",
        "display": "UniFi AP-BaseStationXG"
    },
    "p2N": {
        "base_model": "p2N",
        "display": "PicoStation M2"
    }
};

/**
 * return the full model name for a given device model code
 */
function getDeviceModelName(model) {
    var device_found = device_models[model];
    if (device_found == null) {
        return model;
    }

    return device_models[model].display;
}