<?php
/**
 * UniFi Device Search tool
 *
 * Copyright (c) 2020, Art of WiFi, info@artofwifi.net
 *
 * This source file is part of the UniFi Device Search tool and is subject to the MIT license that is bundled
 * with this package in the file LICENSE.md.
 *
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */

/**
 * load required packages using the composer autoloader together with the file containing shared functions
 */
require_once('../common.php');
require_once('../vendor/autoload.php');

/**
 * load the configuration file if readable
 */
if (is_file('../config/config.php') && is_readable('../config/config.php')) {
    include('../config/config.php');
} else {
    die();
}

/**
 * in order to use the PHP $_SESSION['udst2'] array for temporary storage of variables, session_start() is required
 */
session_start();

/**
 * get the POST parameters that were passed by the calling AJAX function
 */
$site_id        = $_POST['site_id'];
$site_full_name = $_POST['site_full_name'];

/**
 * initialize the results array
 */
$results    = [
    'state'   => 'success',
    'message' => 'controller stats fetched',
    'site'    => [
        'site_id'        => $site_id,
        'site_full_name' => $site_full_name,
    ],
    'timings' => [],
    'data' => [
        'aps'              => [],
        'usws'             => [],
        'usgs'             => [],
        'pending_adoption' => [],
    ]
];

/**
 * and fill in the details for the controller if available
 */
if (!empty($_SESSION['udst2']['controller']) && !empty($_SESSION['udst2']['controller']['idx'])) {
    $time_start = microtime(true);
    $controller = $_SESSION['udst2']['controller'];

    /**
     * create an instance of the Unifi API client class, log in to the controller and pull the requested data
     */
    $unifi_connection = new UniFi_API\Client(trim($controller['user']), trim($controller['password']), trim(rtrim($controller['url'], '/')), $site_id);
    $loginresults     = $unifi_connection->login();

    /**
     * check for login errors
     */
    if ($loginresults === false) {
        $results['state']   = 'error';
        $results['message'] = 'unable to connect to UniFi controller, please check your controller URL and port in <b>config/config.php</b>!';
    } elseif ($loginresults === 400) {
        $results['state']   = 'error';
        $results['message'] = 'UniFi controller login failure, please check your credentials in <b>config/config.php</b>!';
    } else {
        /**
         * store the cookies from the controller for faster reconnects
         */
        $_SESSION['unificookie'] = $unifi_connection->get_cookie();

        /**
         * we can safely continue
         */
        $time_1           = microtime(true);
        $time_after_login = $time_1 - $time_start;

        /**
         * fetch the devices for the selected site and controller
         */
        $devices_array = $unifi_connection->list_devices();

        if (!empty($devices_array)) {
            if ($debug) {
                error_log('UDST DEBUG: site ' . $site_id . ', ' . count($devices_array) . ' devices collected');
            }

            /**
             * then process the devices array
             */
            foreach ($devices_array as $device) {

                switch ($device->state) {
                    case 2:
                        /**
                         * Device is pending adoption so we push it to pending_adoption
                         */
                        $results['data']['pending_adoption'][] = process_device($device);

                        break;
                    default:
                        switch ($device->type) {
                            case 'uap':
                                /**
                                 * push the AP object to the $results['data']['aps'] array
                                 */
                                $results['data']['aps'][] = process_uap($device);
                                break;
                            case 'usw':
                                /**
                                 * push the switch object to the $results['data']['usws'] array
                                 */
                                $results['data']['usws'][] = process_device($device);
                                break;
                            case 'ugw':
                                /**
                                 * push the USG object to the $results['data']['usgs'] array
                                 */
                                $results['data']['usgs'][] = process_device($device);
                                break;
                        }

                        break;
                }
            }
        }

        /**
         * measure timing of data collection from UniFi controller
         */
        $time_2          = microtime(true);
        $time_after_load = $time_2 - $time_start;

        /**
         * calculate all the timings/percentages
         */
        $time_end    = microtime(true);
        $time_total  = $time_end - $time_start;
        $login_perc  = ($time_after_login / $time_total) * 100;
        $load_perc   = (($time_after_load - $time_after_login) / $time_total) * 100;
        $remain_perc = 100 - $login_perc - $load_perc;

        $results['timings']['login'] = $time_after_login;
        $results['timings']['load']  = $time_after_load;
        $results['timings']['login_perc'] = $login_perc;
        $results['timings']['load_perc']  = $load_perc;
    }
} else {
    $results['state']   = 'error';
    $results['message'] = 'unknown controller';
}

/**
 * build the output for switches and routers
 */
function process_device($device) {
    global $site_full_name, $site_id, $unknown_string;

    $extracted_data = [
        'id'        => isset($device->_id) ? $device->_id : $unknown_string,
        'mac'       => isset($device->mac) ? $device->mac : $unknown_string,
        'ip'        => isset($device->ip) ? $device->ip : $unknown_string,
        'name'      => isset($device->name) ? $device->name : $unknown_string,
        'model'     => isset($device->model) ? $device->model : $unknown_string,
        'state'     => isset($device->state) ? $device->state : $unknown_string,
        'version'   => isset($device->version) ? $device->version : $unknown_string,
        'serial'    => isset($device->serial) ? $device->serial : $unknown_string,
        'is_lts'    => isset($device->model_in_lts) ? $device->model_in_lts : false,
        'is_eol'    => isset($device->model_in_eol) ? $device->model_in_eol : false,
        'eol_date'  => isset($device->model_eol_date) ? $device->model_eol_date : '',
        'disabled'  => isset($device->disabled) ? $device->disable : false,
        'site_name' => $site_full_name,
        'site_id'   => $site_id,
    ];
    return $extracted_data;
}

/**
 * build the output for access points
 */
function process_uap($device) {
    global $site_full_name, $site_id, $unknown_string;

    $ssids = [];

    if (!empty($device->vap_table)) {
        foreach ($device->vap_table as $entry) {
            if (!empty($entry->essid) && !in_array($entry->essid, $ssids)) {
                $ssids[] = $entry->essid;
            }
        }
    }

    $extracted_data = [
        'id'        => isset($device->_id) ? $device->_id : $unknown_string,
        'mac'       => isset($device->mac) ? $device->mac : $unknown_string,
        'ip'        => isset($device->ip) ? $device->ip : $unknown_string,
        'name'      => isset($device->name) ? $device->name : $unknown_string,
        'model'     => isset($device->model) ? $device->model : $unknown_string,
        'state'     => isset($device->state) ? $device->state : $unknown_string,
        'version'   => isset($device->version) ? $device->version : $unknown_string,
        'serial'    => isset($device->serial) ? $device->serial : $unknown_string,
        'is_lts'    => isset($device->model_in_lts) ? $device->model_in_lts : false,
        'is_eol'    => isset($device->model_in_eol) ? $device->model_in_eol : false,
        'eol_date'  => isset($device->model_eol_date) ? $device->model_eol_date : '',
        'disabled'  => isset($device->disabled) ? $device->disabled : false,
        'site_name' => $site_full_name,
        'site_id'   => $site_id,
        'ssids'     => $ssids,
    ];

    return $extracted_data;
}

$_SESSION['udst2']['memory_used'] = round(memory_get_peak_usage(false) / 1024 / 1024, 2) . 'M';

if ($debug) {
    error_log('UDST DEBUG: ' . pathinfo($_SERVER['SCRIPT_FILENAME'], PATHINFO_FILENAME) . ', peak memory used is ' . $_SESSION['udst2']['memory_used']);
}

/**
 * output the results with correct JSON formatting
 */
header('Content-Type: application/json; charset=utf-8');
echo (json_encode($results));

ob_end_flush();