<?php
/**
 * UniFi Device Search tool
 *
 * Copyright (c) 2020, Art of WiFi, info@artofwifi.net
 *
 * This source file is part of the UniFi Device Search tool and is subject to the MIT license that is bundled
 * with this package in the file LICENSE.md.
 *
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */

/**
 * load required packages using the composer autoloader together with the file containing shared functions
 */
require_once('../common.php');
require_once('../vendor/autoload.php');

$default_site_id = 'default';

/**
 * load the configuration file if readable
 */
if (is_file('../config/config.php') && is_readable('../config/config.php')) {
    include('../config/config.php');
} else {
    die();
}

/**
 * in order to use the PHP $_SESSION['udst2'] array for temporary storage of variables, session_start() is required
 */
session_start();

/**
 * initialize the results array
 */
$results = [
    'state'   => 'success',
    'message' => 'controller stats fetched',
    'timings' => [],
    'stats'   => [
        'site_count'            => 0,
        'wlan_users_count'      => 0,
        'wlan_guests_count'     => 0,
        'lan_users_count'       => 0,
        'lan_guests_count'      => 0,
        'uap_count'             => 0,
        'ugw_count'             => 0,
        'usw_count'             => 0,
        'pending_devices_count' => 0,
    ],
    'sites'  => [],
];

$pending_devices_count   = 0;
$pending_devices_counted = false;

/**
 * and fill in the details for the controller if available
 */
if (!empty($_SESSION['udst2']['controller']) && !empty($_SESSION['udst2']['controller']['idx'])) {
    $time_start = microtime(true);
    $controller = $_SESSION['udst2']['controller'];

    /**
     * create an instance of the Unifi API client class, log in to the controller and pull the requested data
     */
    $unifi_connection = new UniFi_API\Client(trim($controller['user']), trim($controller['password']), trim(rtrim($controller['url'], '/')), $default_site_id);
    $loginresults     = $unifi_connection->login();

    /**
     * check for login errors
     */
    if ($loginresults === false) {
        $results['state']   = 'error';
        $results['message'] = 'unable to connect to UniFi controller, please check your controller URL and port in <b>config/config.php</b>!';
    } elseif ($loginresults === 400) {
        $results['state']   = 'error';
        $results['message'] = 'UniFi controller login failure, please check your credentials in <b>config/config.php</b>!';
    } else {
        /**
         * store the cookies from the controller for faster reconnects
         */
        $_SESSION['unificookie'] = $unifi_connection->get_cookie();

        $site_info = $unifi_connection->stat_sysinfo();

        if (!empty($site_info) && isset($site_info[0]->version)) {
            $_SESSION['udst2']['controller']['detected_version'] = $site_info[0]->version;
        } else {
            $_SESSION['udst2']['controller']['detected_version'] = 'undetected';
        }

        /**
         * we can safely continue
         */
        $time_1           = microtime(true);
        $time_after_login = $time_1 - $time_start;

        $controller_stats = $unifi_connection->stat_sites();

        if (!empty($controller_stats)) {
            /**
             * site_count
             * wlan_users_count
             * wlan_guests_count
             * lan_users_count
             * lan_guests_count
             * uap_count
             * ugw_count
             * usw_count
             * pending_devices_count
             */
            foreach ($controller_stats as $site) {
                foreach ($site->health as $category) {
                    if ($category->subsystem == 'wlan') {
                        if (property_exists ($category, 'num_user')) {
                            $results['stats']['wlan_users_count'] += $category->num_user;
                        }

                        if (property_exists ($category, 'num_guest')) {
                            $results['stats']['wlan_guests_count'] += $category->num_guest;
                        }

                        if (property_exists ($category, 'num_adopted')) {
                            $results['stats']['uap_count'] += $category->num_adopted;
                        }

                        if (!$pending_devices_counted && property_exists ($category, 'num_pending')) {
                            $pending_devices_count += $category->num_pending;
                        }
                    }

                    if ($category->subsystem == 'wan') {

                        if (property_exists ($category, 'num_adopted')) {
                            $results['stats']['ugw_count'] += $category->num_adopted;
                        }

                        if (!$pending_devices_counted && property_exists ($category, 'num_pending')) {
                            $pending_devices_count += $category->num_pending;
                        }
                    }

                    if ($category->subsystem == 'lan') {
                        if (property_exists ($category, 'num_user')) {
                            $results['stats']['lan_users_count'] += $category->num_user;
                        }

                        if (property_exists ($category, 'num_guest')) {
                            $results['stats']['lan_guests_count'] += $category->num_guest;
                        }

                        if (property_exists ($category, 'num_adopted')) {
                            $results['stats']['usw_count'] += $category->num_adopted;
                        }

                        if (!$pending_devices_counted && property_exists ($category, 'num_pending')) {
                            $pending_devices_count += $category->num_pending;
                        }
                    }
                }

                $results['stats']['site_count']++;

                if (!$pending_devices_counted && $pending_devices_count > 0) {
                    $results['stats']['pending_devices_count'] = $pending_devices_count;
                    $pending_devices_counted = true;
                }

                /**
                 * site is added to the array in the following structure:
                 * - site_id (name)
                 * - site_full_name (desc)
                 */
                $results['sites'][] = [
                    'site_id'        => $site->name,
                    'site_full_name' => $site->desc,
                ];
            }
        }

        /**
         * measure timing of data collection from UniFi controller
         */
        $time_2          = microtime(true);
        $time_after_load = $time_2 - $time_start;

        /**
         * calculate all the timings/percentages
         */
        $time_end    = microtime(true);
        $time_total  = $time_end - $time_start;
        $login_perc  = ($time_after_login / $time_total) * 100;
        $load_perc   = (($time_after_load - $time_after_login) / $time_total) * 100;
        $remain_perc = 100 - $login_perc - $load_perc;

        $results['timings']['login'] = $time_after_login;
        $results['timings']['load']  = $time_after_load;
        $results['timings']['login_perc'] = $login_perc;
        $results['timings']['load_perc']  = $load_perc;
    }
} else {
    $results['state']   = 'error';
    $results['message'] = 'unknown controller';
}

$_SESSION['udst2']['memory_used'] = round(memory_get_peak_usage(false) / 1024 / 1024, 2) . 'M';

if ($debug) {
    error_log(pathinfo('UDST DEBUG: ' . pathinfo($_SERVER['SCRIPT_FILENAME'], PATHINFO_FILENAME), PATHINFO_FILENAME) . ', peak memory used is ' . $_SESSION['udst2']['memory_used']);
}

/**
 * output the results with correct JSON formatting
 */
header('Content-Type: application/json; charset=utf-8');
echo (json_encode($results));

ob_end_flush();